﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/


using System.Collections.Generic;
using System.Reflection;

using Microsoft.Xna.Framework;

using GE.Tools;

namespace GE.Gui
{
    /// <summary>
    /// Enumerates window events.
    /// </summary>
    enum WindowEvent
    {
        Validate,
        Cancel,
        Left,
        Right,
        Up,
        Down
    }

    /// <summary>
    /// A graphic windows.
    /// </summary>
    class Window
    {
        /// <summary>
        /// Owner
        /// </summary>
        protected object m_owner;

        /// <summary>
        /// Position
        /// </summary>
        protected Vector2 m_position;

        /// <summary>
        /// Collection of widgets.
        /// </summary>
        protected List<Widget> m_widgets;

        /// <summary>
        /// Focuse node on the button navigation map.
        /// </summary>
        protected ButtonNode m_focusedNode;

        /// <summary>
        /// Button navigation map.
        /// </summary>
        protected Dictionary<WButton, ButtonNode> m_buttonMap;

        /// <summary>
        /// Links windows actions to callbacks.
        /// </summary>
        protected Dictionary<WindowEvent, MethodInfo> m_actionTable;

        /// <summary>
        /// Owner.
        /// </summary>
        public object Owner
        {
            set
            {
                m_owner = value;
            }
        }

        /// <summary>
        /// Position
        /// </summary>
        public Vector2 Position
        {
            set
            {
                m_position = value;
            }
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public Window()
        {
            m_focusedNode = null;
            m_widgets = new List<Widget>();
            m_position = new Vector2(0f);
            m_buttonMap = new Dictionary<WButton, ButtonNode>();
            m_actionTable = new Dictionary<WindowEvent, MethodInfo>();

            // set default action table
            m_actionTable[WindowEvent.Validate] = GetType().GetMethod("onValidate");
            m_actionTable[WindowEvent.Cancel] = GetType().GetMethod("onClose");
            m_actionTable[WindowEvent.Left] = GetType().GetMethod("onLeft");
            m_actionTable[WindowEvent.Right] = GetType().GetMethod("onRight");
            m_actionTable[WindowEvent.Up] = GetType().GetMethod("onUp");
            m_actionTable[WindowEvent.Down] = GetType().GetMethod("onDown");
        }

        /// <summary>
        /// Show the window.
        /// </summary>
        public void show()
        {
            Gui.Instance.showWindow(this);
            Gui.Instance.pauseInput();
        }

        /// <summary>
        /// Close the window.
        /// </summary>
        public void close()
        {
            Gui.Instance.hideWindow(this);
            Gui.Instance.pauseInput();
        }

        /// <summary>
        /// Add a widget to the window.
        /// </summary>
        /// <param name="widget"></param>
        public void addWidget(Widget widget)
        {
            m_widgets.Add(widget);
        }

        /// <summary>
        /// Register a button
        /// </summary>
        /// <param name="button">button</param>
        /// <param name="left">button on the left</param>
        /// <param name="right">button on the right</param>
        /// <param name="top">button on the top</param>
        /// <param name="bottom">button on the bottom</param>
        public void registerButton(WButton button, WButton left, WButton right,
            WButton top, WButton bottom)
        {
            ButtonNode node = new ButtonNode(button, left, right, top, bottom);
            m_buttonMap[button] = node;
            if (m_focusedNode == null) // focus first button
            {
                m_focusedNode = node;
                m_focusedNode.Button.IsFocused = true;
            }
        }

        /// <summary>
        /// Call the the windows receive the Validate event
        /// </summary>
        public virtual void onValidate()
        {
            m_focusedNode.Button.click();
        }

        /// <summary>
        /// Call the the windows receive the Cancer event.
        /// </summary>
        public virtual void onClose()
        {
            close();
        }

        /// <summary>
        /// Call the the windows receive the Left event.
        /// </summary>
        public virtual void onLeft()
        {
            if (m_focusedNode != null)
            {
                if (m_focusedNode.Left != null)
                {
                    m_focusedNode.Button.IsFocused = false;
                    m_focusedNode = m_buttonMap[m_focusedNode.Left];
                    m_focusedNode.Button.IsFocused = true;
                    Gui.Instance.pauseInput();
                }
            }
        }

        /// <summary>
        /// Call the the windows receive the Right event.
        /// </summary>
        public virtual void onRight()
        {
            if (m_focusedNode != null)
            {
                if (m_focusedNode.Right != null)
                {
                    m_focusedNode.Button.IsFocused = false;
                    m_focusedNode = m_buttonMap[m_focusedNode.Right];
                    m_focusedNode.Button.IsFocused = true;
                    Gui.Instance.pauseInput();
                }
            }
        }

        /// <summary>
        /// Call the the windows receive the Up event.
        /// </summary>
        public virtual void onUp()
        {
            if (m_focusedNode != null)
            {
                if (m_focusedNode.Top != null)
                {
                    m_focusedNode.Button.IsFocused = false;
                    m_focusedNode = m_buttonMap[m_focusedNode.Top];
                    m_focusedNode.Button.IsFocused = true;
                    Gui.Instance.pauseInput();
                }
            }
        }

        /// <summary>
        /// Call the the windows receive the Down event.
        /// </summary>
        public virtual void onDown()
        {
            if (m_focusedNode != null)
            {
                if (m_focusedNode.Bottom != null)
                {
                    m_focusedNode.Button.IsFocused = false;
                    m_focusedNode = m_buttonMap[m_focusedNode.Bottom];
                    m_focusedNode.Button.IsFocused = true;
                    Gui.Instance.pauseInput();
                }
            }
        }

        /// <summary>
        /// Render the window.
        /// </summary>
        public virtual void render()
        {
            foreach (Widget w in m_widgets)
                w.render(m_position);
        }

        /// <summary>
        /// Update the window.
        /// </summary>
        public virtual void update()
        {
            foreach (Widget w in m_widgets)
                w.update();
        }

        /// <summary>
        ///  Send an event to the window.
        /// </summary>
        /// <param name="action"></param>
        public virtual void sendEvent(WindowEvent action)
        {
            try
            {
                m_actionTable[action].Invoke(this, new object[] { });
            }
            catch (KeyNotFoundException e)
            {
                Logger.Instance.error(this, e, "Action not mapped");
            }
            catch(TargetInvocationException e)
            {
                Logger.Instance.error(this, e, "Error in event callback");
            }
        }
    }
}
